﻿using System;
using System.IO;
using BobsPizzaPlace.BO;
using Habanero.Base;
using Habanero.BO;
using Habanero.Console;

namespace BobsPizzaPlace.Console
{
    class Program
    {
        private static HabaneroAppConsoleInMemory _habaneroApp;

        static void Main(string[] args)
        {
            _habaneroApp = new HabaneroAppConsoleInMemory("Bob's Pizza Place", "0.1");
            _habaneroApp.Startup();

            LoadDataStoreFromFile();

            BusinessObjectCollection<Pizza> pizzas = Broker.GetBusinessObjectCollection<Pizza>("", "Name");

            DisplayMenu(pizzas);

            string input = System.Console.ReadLine().ToUpper();
            while (input != "Q")
            {
                if (input == "A")
                {
                    AddNewPizza();
                    pizzas.Refresh();
                }
                else if (input == "D")
                {
                    System.Console.Out.WriteLine("Which pizza would you like to delete? ");
                    int pizzaNumber = Convert.ToInt32(System.Console.ReadLine());
                    DeletePizza(pizzas[pizzaNumber]);
                } else if (input == "O")
                {
                    CaptureOrder(pizzas);
                }
                else
                {
                    int pizzaNumber = Convert.ToInt32(input);
                    EditPizza(pizzas[pizzaNumber]);
                }
                DisplayMenu(pizzas);
                input = System.Console.ReadLine().ToUpper();
            }

            WriteDataStoreToFile();
        }

        private static void CaptureOrder(BusinessObjectCollection<Pizza> pizzas)
        {
            Order order = new Order();
            CaptureOrderItemsForOrder(pizzas, order);
            order.Save();
            DisplayOrder(order);
        }

        private static void DisplayOrder(Order order)
        {
            System.Console.Out.WriteLine("Order Number: " + order.OrderNumber);
            order.OrderItems.ForEach(
                item =>
                System.Console.Out.WriteLine("Pizza: {0}, Number: {1}, Subtotal: {2:0.00}", item.Pizza.Name,
                                             item.NumberOfItems, item.Subtotal));
            System.Console.Out.WriteLine("Total: {0:0.00}", order.Total);
        }

        private static void CaptureOrderItemsForOrder(BusinessObjectCollection<Pizza> pizzas, Order order)
        {
            do
            {
                System.Console.Out.WriteLine("Select a pizza (or 'x' to complete order):");
                int i = 0;
                pizzas.ForEach(pizza =>
                               System.Console.Out.WriteLine("{0}. {1} {2:0.00}",
                                                            i++, pizza.Name, pizza.Price));

                string input = System.Console.ReadLine();
                if (input.ToUpper() == "X") return;

                int numberSelected = Convert.ToInt32(input);
                Pizza selectedPizza = pizzas[numberSelected];
                System.Console.Out.WriteLine("{0} selected. How many?", selectedPizza.Name);
                int numberOfPizzas = Convert.ToInt32(System.Console.ReadLine());
                OrderItem orderItem = new OrderItem {Pizza = selectedPizza, NumberOfItems = numberOfPizzas};
                order.OrderItems.Add(orderItem);
            } while (true); 
        }

        private static void DeletePizza(Pizza pizza)
        {
            pizza.MarkForDelete();
            pizza.Save();
        }

        private static void EditPizza(Pizza pizza)
        {
            System.Console.Out.WriteLine("Updating price of {0} pizza", pizza.Name);
            System.Console.Out.WriteLine("Old price: {0:0.00}", pizza.Price);
            System.Console.Out.Write("New price: ");
            pizza.Price = Convert.ToDecimal(System.Console.ReadLine());
            pizza.Save();
            System.Console.Out.WriteLine("Price updated");
        }

        private static void LoadDataStoreFromFile()
        {
            if (!File.Exists("pizzas.xml")) WriteDataStoreToFile();
            using (Stream pizzasStream = File.OpenRead("pizzas.xml"))
            {
                DataStoreInMemoryXmlReader reader =
                    new DataStoreInMemoryXmlReader(pizzasStream);
                _habaneroApp.DataStoreInMemory.AllObjects = reader.Read();
            }
        }

        private static void WriteDataStoreToFile()
        {
            using (Stream pizzasStream = File.Create("pizzas.xml"))
            {
                DataStoreInMemoryXmlWriter writer =
                    new DataStoreInMemoryXmlWriter(pizzasStream);
                writer.Write(_habaneroApp.DataStoreInMemory);
            }
        }

        private static void AddNewPizza()
        {
            Pizza newPizza = new Pizza();
            System.Console.Out.WriteLine("New Pizza");
            System.Console.Out.Write("Name: ");
            newPizza.Name = System.Console.ReadLine();
            System.Console.Out.Write("Price: ");
            newPizza.Price = Convert.ToDecimal(System.Console.ReadLine());
            try
            {
                newPizza.Save();
            }
            catch (BusinessObjectException ex)
            {
                System.Console.Out.WriteLine(ex.Message);
            }
        }

        private static void DisplayMenu(BusinessObjectCollection<Pizza> pizzas)
        {
            int i = 0;
            pizzas.ForEach(pizza =>
                           System.Console.Out.WriteLine("{0}. {1} {2:0.00}",
                                                        i++, pizza.Name, pizza.Price));

            System.Console.Out.WriteLine("A. Add a new pizza");
            System.Console.Out.WriteLine("D. Delete a pizza");
            System.Console.Out.WriteLine("O. Capture an order");
            System.Console.Out.WriteLine("Q. Quit");
        }
    }
}
